<?php

namespace App\Http\Controllers;

use Barryvdh\DomPDF;
use Barryvdh\DomPDF\PDF;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class PagoController extends Controller
{

    public function index() {}


    public function create() {}

    public function store(Request $request)
    {
        $request->validate([
            "txtidaporte" => "required",
            "txtidpadre" => "required",
            "txtidusuario" => "required",
            "txtmontoaportar" => "required",
            "txtmontoaportado" => "required",
            "txtmontodebe" => "required",
            "txtfechapago" => "required",
            //"txtnumerorecibo" => "required",
            //"txtfotoboucher" =>  "required|image|mimes:jpeg,png,jpg,pdf",
        ]);

        $id_aporte = $request->txtidaporte;
        $id_padre = $request->txtidpadre;
        $id_usuario = $request->txtidusuario;
        $monto_aportado = $request->txtmontoaportado;
        $monto_debe = $request->txtmontodebe;

        //actualizacion
        $fecha_pago = $request->txtfechapago;
        $numero_recibo = $request->txtnumerorecibo;

        //verificar que $numero_recibo no este registrado en la tabla pago
        $verificarRecibo = DB::select(" SELECT
            pago.numero_pago_boucher,
            pago.id_padre_familia,
            concat(
            padre_familia.padre_dni,' - ',
            padre_familia.padre_nombres, ' ',
            padre_familia.padre_ape_pat,' ',
            padre_familia.padre_ape_mat) as padre_familia
            FROM
            pago
            INNER JOIN padre_familia ON pago.id_padre_familia = padre_familia.id_padre_familia
            WHERE numero_pago_boucher= ?", [$numero_recibo]);
        if ($verificarRecibo != null) {
            return back()->with("INCORRECTO", "El numero de operacion/boucher ya esta registrado por el padre de familia: " . $verificarRecibo[0]->padre_familia);
        }

        //verificar que el padre de familia con el id_aporte este registrado en la tabla aporte_padres
        $verificar = DB::select(
            "SELECT
            aporte_padres.id_aporte_padres,
            aporte_padres.id_aporte,
            aporte_padres.id_padre_familia,
            aporte_padres.monto_aporte,
            aporte_padres.monto_aportado,
            aporte_padres.monto_aporte - aporte_padres.monto_aportado as debe,
            aporte.monto
            FROM
            aporte_padres
            INNER JOIN aporte ON aporte_padres.id_aporte = aporte.id_aporte
            WHERE aporte_padres.id_aporte=? AND id_padre_familia=?
            ",
            [$id_aporte, $id_padre]
        );
        if ($verificar == null) {
            return back()->with("INCORRECTO", "El padre de familia no esta registrado en la tabla aporte_padres");
        }
        if ($verificar[0]->debe != null && $verificar[0]->debe == 0) {
            return back()->with("INCORRECTO", "El padre de familia ya cancelo el aporte");
        }

        // Obtenemos el id_aporte_padres
        $id_aporte_padres = $verificar[0]->id_aporte_padres;

        //actualizar el monto aportado y el monto debe en la tabla aporte_padres
        $actualizar = DB::update(
            "UPDATE aporte_padres SET monto_aportado = COALESCE(monto_aportado, 0) + COALESCE(?, 0), debe=? WHERE id_aporte=? AND id_padre_familia=?",
            [$monto_aportado, $monto_debe, $id_aporte, $id_padre]
        );


        // si existe la foto del boucher subir a la carpeta storage/app/public/ARCHIVOS/pagos. el nombre del archivo debe contener la fecha y numero de boucher
        $nombre_foto = null;
        if ($request->hasFile('txtfotoboucher')) {
            $foto_boucher = $request->file('txtfotoboucher');
            $nombre_foto = $fecha_pago . "-" . $numero_recibo . "." . $foto_boucher->getClientOriginalExtension();
            $foto_boucher->storeAs('public/ARCHIVOS/pagos', $nombre_foto);
        }


        //registrar pago en la tabla pago
        $tituloAporte = DB::select("SELECT titulo FROM aporte WHERE id_aporte=?", [$id_aporte]);
        $titulo = $tituloAporte[0]->titulo;
        $registrar = DB::insert(
            "INSERT INTO pago(id_padre_familia, id_usuario, pago_concepto, monto_pago, debe, fecha_pago_boucher, numero_pago_boucher,foto_boucher,comentario, id_aporte_padres, estado) VALUES(?,?,?,?,?,?,?,?,?,?,?)",
            [$id_padre, $id_usuario, "APORTE: $titulo", $monto_aportado, $monto_debe, $fecha_pago, $numero_recibo, $nombre_foto, $request->txtcomentarios, $id_aporte_padres, 'VIGENTE']
        );



        if ($actualizar == 1 && $registrar == 1) {
            $id_pago = DB::getPdo()->lastInsertId();

            // Registrar en historial_pagos
            $usuario = auth()->user()->nombres;
            $descripcion = "El usuario '{$usuario}' ha registrado un pago por el aporte '{$titulo}' con el número de operación {$numero_recibo} por un monto de S/. " . number_format($monto_aportado, 2);
            try {
                DB::insert("INSERT INTO historial_pagos(id_padre_familia, descripcion, registrado_por, fecha) VALUES (?, ?, ?, ?)", [$id_padre, $descripcion, $usuario, now()]);
            } catch (\Throwable $th) {
                //throw $th;
            }

            return back()->with([
                "id_pago" => $id_pago
            ]);
        } else {
            return back()->with("INCORRECTO", "El pago no se registro correctamente");
        }
    }

    //ultima actualizacion
    public function editarPago(Request $request)
    {
        $request->validate([
            "txtidaporte" => "required",
            "txtidpadre" => "required",
            "txtidusuario" => "required",
            "txtmontoaportar" => "required",
            "txtcomentarios" => "required",
        ]);

        $id_aporte = $request->txtidaporte;
        $id_padre = $request->txtidpadre;
        $monto_aportar = $request->txtmontoaportar;
        $comentarios = $request->txtcomentarios;

        //verificar que el padre de familia con el id_aporte este registrado en la tabla aporte_padres
        $verificar = DB::select(
            "SELECT
            aporte_padres.id_aporte_padres,
            aporte_padres.id_aporte,
            aporte_padres.id_padre_familia,
            aporte_padres.monto_aporte,
            aporte_padres.monto_aportado,
            aporte_padres.monto_aporte - aporte_padres.monto_aportado as debe,
            aporte.monto
            FROM
            aporte_padres
            INNER JOIN aporte ON aporte_padres.id_aporte = aporte.id_aporte
            WHERE aporte_padres.id_aporte=? AND id_padre_familia=?
            ",
            [$id_aporte, $id_padre]
        );
        if ($verificar == null) {
            return back()->with("INCORRECTO", "El padre de familia no esta registrado en la tabla aporte_padres");
        }
        if ($verificar[0]->debe != null && $verificar[0]->debe == 0) {
            return back()->with("INCORRECTO", "El padre de familia ya cancelo el aporte");
        }

        //actualizar el monto 
        $actualizar = DB::update(
            "UPDATE aporte_padres SET monto_aporte = ?, monto_aportado=?, debe=?, comentario=? WHERE id_aporte=? AND id_padre_familia=?",
            [$monto_aportar, "0", null, $comentarios, $id_aporte, $id_padre]
        );

        // registrar historial de pagos
        $usuario = auth()->user()->nombres;
        $descripcion = "El usuario '{$usuario}' ha editado el pago del aporte con ID {$id_aporte} de monto S/. {$verificar[0]->monto_aporte} a un monto de S/. {$monto_aportar}. Comentario: {$comentarios}";
        try {
            DB::insert("INSERT INTO historial_pagos(id_padre_familia, descripcion, registrado_por, fecha) VALUES (?, ?, ?, ?)", [$id_padre, $descripcion, $usuario, now()]);
        } catch (\Throwable $th) {
            //throw $th;
        }


        if ($actualizar == 1) {
            return back()->with("CORRECTO", "El monto del aporte se actualizo correctamente");
        } else {
            return back()->with("INCORRECTO", "El pago no se registro correctamente");
        }
    }

    public function pagarMultaReunion(Request $request)
    {
        //return $request;
        $request->validate([
            "txtidreunion" => "required",
            "txtidpadre" => "required",
            "txtidusuario" => "required",
            "txtmulta" => "required",
            "txtpago" => "required",
            "txtfechapago" => "required",
            "txtnumerorecibo" => "nullable",
            "txtcomentarios" => "nullable",
        ]);

        //verificar que la columna detalles de la tabla reunion_padres este vacia
        $verificar = DB::select(
            "SELECT
            reunion_padres.id_reunion_padres,
            reunion_padres.detalles,
            reunion_padres.id_reunion,
            reunion_padres.id_padre_familia,
            reunion_padres.asistencia
            FROM
            reunion_padres
            WHERE reunion_padres.id_reunion=? AND reunion_padres.id_padre_familia=?",
            [$request->txtidreunion, $request->txtidpadre]
        );
        if ($verificar[0]->asistencia == null and $verificar[0]->detalles != null) {
            return back()->with("INCORRECTO", "El padre de familia ya cancelo la multa");
        }

        // Procesar la imagen si existe
        $nombre_foto = null;
        if ($request->hasFile('txtfotoboucher')) {
            $foto = $request->file('txtfotoboucher');
            $nombre_foto = time() . '_' . $foto->getClientOriginalName();
            $foto->storeAs('ARCHIVOS/pagos', $nombre_foto, 'public');
        }

        // Obtenemos el id_reunion_padres
        $id_reunion_padres = $verificar[0]->id_reunion_padres;

        //actualizar la columna detalles de la tabla reunion_padres
        $actualizar = DB::update(
            "UPDATE reunion_padres SET detalles=?, monto_aportado=? WHERE id_reunion=? AND id_padre_familia=? ",
            ["MULTA PAGADA", $request->txtpago, $request->txtidreunion, $request->txtidpadre]
        );

        //registrar pago en la tabla pago
        $tituloReunion = DB::select("SELECT titulo FROM reunion WHERE id_reunion=?", [$request->txtidreunion]);
        $titulo = $tituloReunion[0]->titulo;
        $fecha_pago = $request->txtfechapago;
        $numero_recibo = $request->txtnumerorecibo;
        $comentarios = $request->txtcomentarios;

        $registrar = DB::insert(
            "INSERT INTO pago(id_padre_familia, id_usuario, pago_concepto, monto_pago, debe, fecha_pago_boucher, numero_pago_boucher, foto_boucher, comentario, id_reunion_padres, estado) VALUES(?,?,?,?,?,?,?,?,?,?,?)",
            [$request->txtidpadre, $request->txtidusuario, "MULTA REUNION: $titulo", $request->txtpago, 0, $fecha_pago, $numero_recibo, $nombre_foto, $comentarios, $id_reunion_padres, 'VIGENTE']
        );

        if ($actualizar == 1 && $registrar == 1) {
            //ultimo id
            $id_pago = DB::getPdo()->lastInsertId();

            // Registrar en historial_pagos
            $usuario = auth()->user()->nombres;
            $descripcion = "El usuario '{$usuario}' ha registrado un pago de multa por la reunión '{$titulo}' con el número de operación {$numero_recibo} por un monto de S/. " . number_format($request->txtpago, 2);
            try {
                DB::insert("INSERT INTO historial_pagos(id_padre_familia, descripcion, registrado_por, fecha) VALUES (?, ?, ?, ?)", [$request->txtidpadre, $descripcion, $usuario, now()]);
            } catch (\Throwable $th) {
                //throw $th;
            }

            return back()->with([
                "id_pago" => $id_pago
            ]);
        } else {
            return back()->with("INCORRECTO", "Error al registrar el pago");
        }
    }


    public function show($dni_padre)
    {
        $nombrePadreFamilia = DB::select("select concat(padre_nombres,' ',padre_ape_pat,' ', padre_ape_mat) as nombre, id_padre_familia from padre_familia where padre_dni=?", [$dni_padre]);
        if ($nombrePadreFamilia == null) {
            return back()->with("INCORRECTO", "El padre de familia no existe");
        }
        $id_padre = $nombrePadreFamilia[0]->id_padre_familia;
        $cantidadHijos = DB::select("SELECT COUNT(*) as cantidad FROM estudiante WHERE id_padre_familia=?", [$id_padre]);

        $historialAportes = DB::select(
            "SELECT
            ap.id_aporte_padres,
            ap.id_aporte,
            ap.id_padre_familia,
            ap.monto_aportado,
            ap.comentario,
            aporte.titulo,
            aporte.descripcion,
            aporte.monto,
            aporte.fecha,
            ap.monto_aporte,
            (ap.monto_aporte - ap.monto_aportado) as debe
            FROM
            aporte_padres AS ap
            INNER JOIN aporte ON ap.id_aporte = aporte.id_aporte
            WHERE
                ap.id_padre_familia = ?
            GROUP BY
                ap.id_aporte_padres, ap.id_aporte, ap.id_padre_familia, aporte.titulo, 
                aporte.descripcion, aporte.monto, aporte.fecha
            ORDER BY debe DESC",
            [$id_padre]
        );
        //return $historialAportes;

        $historialReuniones = DB::select(
            " SELECT
            reunion_padres.*,
            reunion.titulo,
            reunion.id_estado_reunion,
            reunion.descripcion,
            reunion.multa_precio,
            reunion.fecha,
            reunion.hora,
            estado_reunion.estado
            FROM
            reunion_padres
            INNER JOIN reunion ON reunion_padres.id_reunion = reunion.id_reunion
            INNER JOIN estado_reunion ON reunion.id_estado_reunion = estado_reunion.id_estado_reunion        
            where id_padre_familia=? order by reunion.fecha desc ",
            [$id_padre]
        );

        $historialPagos = DB::select(
            "SELECT
            pago.id_pago,
            pago.id_padre_familia,
            pago.id_usuario,
            pago.pago_concepto,
            pago.monto_pago,
            pago.fecha,
            pago.foto_boucher,
            pago.comentario,
            pago.estado,
            usuario.nombres as nombres_usuario
            FROM
            pago
            INNER JOIN usuario ON pago.id_usuario = usuario.id_usuario
            WHERE pago.id_padre_familia = ?
            ORDER BY pago.fecha DESC",
            [$id_padre]
        );

        // Obtener el historial de modificaciones de pagos
        $historialModificaciones = DB::select(
            "SELECT
            hp.id_historial_pago,
            hp.descripcion,
            hp.registrado_por,
            hp.fecha as fecha
            FROM
            historial_pagos hp
            WHERE hp.id_padre_familia = ?
            ORDER BY hp.fecha DESC",
            [$id_padre]
        );
        
        return view("vistas/pagos.index", compact("historialAportes", "historialReuniones", "nombrePadreFamilia"))
            ->with("cantidadHijos", $cantidadHijos[0]->cantidad)
            ->with("historialPagos", $historialPagos)
            ->with("dni_padre", $dni_padre)
            ->with("historialModificaciones", $historialModificaciones)
            ->with("id_padre_familia", $id_padre);
    }

    public function ticketPago($id_pago)
    {
        $pagoDetalle = DB::select(" SELECT
        usuario.nombres,
        padre_familia.padre_nombres,
        padre_familia.padre_dni,
        padre_familia.padre_ape_pat,
        padre_familia.padre_ape_mat,
        pago.id_pago,
        pago.monto_pago,
        pago.id_padre_familia,
        pago.pago_concepto,
        pago.comentario
        FROM
        pago
        INNER JOIN padre_familia ON pago.id_padre_familia = padre_familia.id_padre_familia
        INNER JOIN usuario ON pago.id_usuario = usuario.id_usuario
        where id_pago=$id_pago ");

        $empresa = DB::select("SELECT * FROM empresa");
        $id_padre = $pagoDetalle[0]->id_padre_familia;

        $pdf = \App::make('dompdf.wrapper');
        //tamaño ticket
        $pdf->setPaper(array(0, 0, 226, 500));
        $pdf->loadView('vistas/pagos.ticket', compact('pagoDetalle', 'empresa', "id_padre"));
        return $pdf->stream("ticket_pago-$id_pago.pdf");
    }


    public function actualizarBoucher(Request $request)
    {
        $request->validate([
            'id_pago' => 'required|exists:pago,id_pago',
            'nueva_foto_boucher' => 'required|file|mimes:jpeg,png,jpg,pdf',
        ]);

        try {
            // Obtener el registro de pago actual
            $pago = DB::select("SELECT * FROM pago WHERE id_pago = ?", [$request->id_pago]);

            if (empty($pago)) {
                return back()->with('INCORRECTO', 'El pago no existe');
            }

            $pago = $pago[0];

            // Obtener la imagen anterior
            $imagenAnterior = $pago->foto_boucher;

            // Procesar la nueva imagen
            $archivo = $request->file('nueva_foto_boucher');
            $nombreArchivo = time() . '_' . $archivo->getClientOriginalName();

            // Guardar la nueva imagen
            $archivo->storeAs('public/ARCHIVOS/pagos', $nombreArchivo);

            // Actualizar el registro en la base de datos
            $actualizar = DB::update(
                "UPDATE pago SET foto_boucher = ? WHERE id_pago = ?",
                [$nombreArchivo, $request->id_pago]
            );

            // Eliminar la imagen anterior si existe
            if ($imagenAnterior && $actualizar) {
                $rutaImagenAnterior = storage_path('app/public/ARCHIVOS/pagos/' . $imagenAnterior);
                if (file_exists($rutaImagenAnterior)) {
                    unlink($rutaImagenAnterior);
                }
            }

            if ($actualizar) {
                return back()->with('CORRECTO', 'Comprobante actualizado correctamente');
            } else {
                return back()->with('INCORRECTO', 'Error al actualizar el comprobante');
            }
        } catch (\Exception $e) {
            return back()->with('INCORRECTO', 'Error: ' . $e->getMessage());
        }
    }



    public function ticketPagos(Request $request)
    {
        $id_pagos = $request->input('id_pagos');
        $comentarios = $request->input("comentarios");

        // Decodificar la cadena JSON en un array
        $id_pagos = json_decode($id_pagos, true);
        $comentarios = json_decode($comentarios, true);

        if (empty($id_pagos) || !is_array($id_pagos)) {
            return back()->with("INCORRECTO", "No se proporcionaron IDs de pagos válidos.");
        }

        $pagos = DB::table('pago')
            ->join('padre_familia', 'pago.id_padre_familia', '=', 'padre_familia.id_padre_familia')
            ->join('usuario', 'pago.id_usuario', '=', 'usuario.id_usuario')
            ->whereIn('pago.id_pago', $id_pagos)
            ->select(
                'usuario.nombres',
                'padre_familia.padre_nombres',
                'padre_familia.padre_dni',
                'padre_familia.padre_ape_pat',
                'padre_familia.padre_ape_mat',
                'pago.id_pago',
                'pago.monto_pago',
                'pago.id_padre_familia',
                'pago.pago_concepto'
            )
            ->get();

        if ($pagos->isEmpty()) {
            return back()->with("INCORRECTO", "No se encontraron pagos con los IDs proporcionados.");
        }

        $empresa = DB::select("SELECT * FROM empresa");
        $id_padre = $pagos->first()->id_padre_familia;

        $pdf = \App::make('dompdf.wrapper');
        // tamaño ticket
        $pdf->setPaper(array(0, 0, 200, 500));
        $pdf->loadView('vistas/pagos.ticketPagos', compact('pagos', 'empresa', 'id_padre', 'comentarios'));
        return $pdf->stream("ticket_pagos.pdf");
    }

    public function registrarPagos(Request $request)
    {
        try {
            // Validación de datos
            $validatedData = $request->validate([
                'aportes' => 'required|array',
                'aportes.*.id_aporte' => 'required|numeric',
                'aportes.*.monto' => 'required|numeric|min:0',
                'id_padre_familia' => 'required|numeric',
                'comentarios' => 'nullable|string',
                'fecha_pago' => 'required|date',
                'numero_recibo' => 'nullable|string',
                'foto_boucher' => 'nullable|file|mimes:jpeg,png,jpg,pdf',
            ]);

            $id_padre = $request->id_padre_familia;
            $id_usuario = auth()->user()->id_usuario;
            $comentarios = $request->comentarios;
            $fecha_pago = $request->fecha_pago;
            $numero_recibo = $request->numero_recibo;
            $id_pagos = [];

            // Verificar que el número de recibo no esté ya registrado (solo si se proporciona)
            if ($numero_recibo) {
                $verificarRecibo = DB::select("SELECT
                    pago.numero_pago_boucher,
                    pago.id_padre_familia,
                    concat(
                    padre_familia.padre_dni,' - ',
                    padre_familia.padre_nombres, ' ',
                    padre_familia.padre_ape_pat,' ',
                    padre_familia.padre_ape_mat) as padre_familia
                    FROM
                    pago
                    INNER JOIN padre_familia ON pago.id_padre_familia = padre_familia.id_padre_familia
                    WHERE numero_pago_boucher = ?", [$numero_recibo]);

                if ($verificarRecibo != null) {
                    return back()->with("INCORRECTO", "El número de operación/boucher ya está registrado por el padre de familia: " . $verificarRecibo[0]->padre_familia);
                }
            }

            DB::beginTransaction();

            // Procesar el archivo del boucher si existe
            $nombre_foto = null;
            if ($request->hasFile('foto_boucher')) {
                $foto_boucher = $request->file('foto_boucher');
                $nombre_foto = $fecha_pago . "-" . ($numero_recibo ?: time()) . "." . $foto_boucher->getClientOriginalExtension();
                $foto_boucher->storeAs('public/ARCHIVOS/pagos', $nombre_foto);
            }

            foreach ($request->aportes as $id_aporte => $aporte) {
                $id_aporte = $aporte['id_aporte'];
                $monto_aportado = $aporte['monto'];

                // Verificar que el aporte existe y el padre está registrado
                $verificar = DB::select(
                    "SELECT
                    aporte_padres.id_aporte_padres,
                    aporte_padres.id_aporte,
                    aporte_padres.id_padre_familia,
                    aporte_padres.monto_aporte,
                    aporte_padres.monto_aportado,
                    aporte_padres.monto_aporte - aporte_padres.monto_aportado as debe,
                    aporte.titulo,
                    aporte.monto
                    FROM
                    aporte_padres
                    INNER JOIN aporte ON aporte_padres.id_aporte = aporte.id_aporte
                    WHERE aporte_padres.id_aporte=? AND id_padre_familia=?
                    ",
                    [$id_aporte, $id_padre]
                );

                if (empty($verificar)) {
                    DB::rollBack();
                    return back()->with("INCORRECTO", "Error: Uno de los aportes seleccionados no está registrado para este padre de familia");
                }

                $debe = $verificar[0]->debe;
                $titulo = $verificar[0]->titulo;
                $id_aporte_padres = $verificar[0]->id_aporte_padres;

                if ($debe <= 0) {
                    DB::rollBack();
                    return back()->with("INCORRECTO", "Error: El aporte {$titulo} ya está pagado");
                }

                if ($monto_aportado > $debe) {
                    $monto_aportado = $debe; // Ajustar si el monto es mayor a la deuda
                }

                $nuevo_debe = $debe - $monto_aportado;

                // Actualizar el monto aportado y la deuda en aporte_padres
                $updateResult = DB::update(
                    "UPDATE aporte_padres SET 
                    monto_aportado = COALESCE(monto_aportado, 0) + ?, 
                    debe = ?
                    WHERE id_aporte = ? AND id_padre_familia = ?",
                    [$monto_aportado, $nuevo_debe, $id_aporte, $id_padre]
                );

                if (!$updateResult) {
                    DB::rollBack();
                    return back()->with("INCORRECTO", "Error al actualizar el aporte {$titulo}");
                }

                // Registrar el pago en la tabla pago
                $insertResult = DB::insert(
                    "INSERT INTO pago(id_padre_familia, id_usuario, pago_concepto, monto_pago, debe, fecha_pago_boucher, numero_pago_boucher, foto_boucher, comentario, id_aporte_padres, estado) 
                    VALUES(?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)",
                    [$id_padre, $id_usuario, "APORTE: $titulo", $monto_aportado, $nuevo_debe, $fecha_pago, $numero_recibo, $nombre_foto, $comentarios, $id_aporte_padres, 'VIGENTE']
                );

                if (!$insertResult) {
                    DB::rollBack();
                    return back()->with("INCORRECTO", "Error al registrar el pago para {$titulo}");
                }

                // Guardar el ID del pago
                $id_pagos[] = DB::getPdo()->lastInsertId();
            }

            // Registrar en historial_pagos
            $usuario = auth()->user()->nombres;
            $totalPagado = array_sum(array_map(function ($id) {
                $pago = DB::select("SELECT monto_pago FROM pago WHERE id_pago = ?", [$id]);
                return $pago[0]->monto_pago ?? 0;
            }, $id_pagos));

            $descripcion = "El usuario '{$usuario}' ha registrado " . count($id_pagos) . " pagos múltiples" .
                ($numero_recibo ? " con el número de operación {$numero_recibo}" : "") .
                " por un total de S/. " . number_format($totalPagado, 2);

            DB::insert(
                "INSERT INTO historial_pagos (id_padre_familia, descripcion, registrado_por, fecha) VALUES (?, ?, ?, ?)",
                [$id_padre, $descripcion, $usuario, now()]
            );

            DB::commit();

            // Array asociativo para mapear cada id_pago con su comentario específico
            $comentarios_array = array_fill_keys($id_pagos, $comentarios);

            // Guardar un mensaje de éxito en la sesión
            return redirect()->back()->with([
                'id_pagos' => $id_pagos,
                'comentarios' => $comentarios_array,
                //'CORRECTO' => 'Pagos múltiples registrados correctamente'
            ]);
        } catch (\Illuminate\Validation\ValidationException $e) {
            // Manejar errores de validación
            return back()->withErrors($e->validator)->withInput()->with("INCORRECTO", "Por favor corrija los errores en el formulario");
        } catch (\Exception $e) {
            // Manejar cualquier otra excepción
            DB::rollBack();
            return back()->with("INCORRECTO", "Error al registrar los pagos: " . $e->getMessage())->withInput();
        }
    }

    public function registrarPagosMultas(Request $request)
    {
        try {
            // Validación de datos
            $validatedData = $request->validate([
                'multas' => 'required|array',
                'multas.*.id_reunion' => 'required|numeric',
                'multas.*.monto' => 'required|numeric|min:0',
                'id_padre_familia' => 'required|numeric',
                'comentarios' => 'nullable|string',
                'fecha_pago' => 'required|date',
                'numero_recibo' => 'nullable|string',
                'foto_boucher' => 'nullable|file|mimes:jpeg,png,jpg,pdf',
            ]);

            $id_padre = $request->id_padre_familia;
            $id_usuario = auth()->user()->id_usuario;
            $comentarios = $request->comentarios;
            $fecha_pago = $request->fecha_pago;
            $numero_recibo = $request->numero_recibo;
            $id_pagos = [];

            // Verificar que el número de recibo no esté ya registrado (solo si se proporciona)
            if ($numero_recibo) {
                $verificarRecibo = DB::select("SELECT
                    pago.numero_pago_boucher,
                    pago.id_padre_familia,
                    CONCAT(
                    padre_familia.padre_dni,' - ',
                    padre_familia.padre_nombres, ' ',
                    padre_familia.padre_ape_pat,' ',
                    padre_familia.padre_ape_mat) as padre_familia
                    FROM
                    pago
                    INNER JOIN padre_familia ON pago.id_padre_familia = padre_familia.id_padre_familia
                    WHERE numero_pago_boucher = ?", [$numero_recibo]);

                if ($verificarRecibo != null) {
                    return back()->with("INCORRECTO", "El número de operación/boucher ya está registrado por el padre de familia: " . $verificarRecibo[0]->padre_familia);
                }
            }

            DB::beginTransaction();

            // Procesar el archivo del boucher si existe
            $nombre_foto = null;
            if ($request->hasFile('foto_boucher')) {
                $foto_boucher = $request->file('foto_boucher');
                $nombre_foto = $fecha_pago . "-" . ($numero_recibo ?: time()) . "." . $foto_boucher->getClientOriginalExtension();
                $foto_boucher->storeAs('public/ARCHIVOS/pagos', $nombre_foto);
            }

            foreach ($request->multas as $id_reunion => $multa) {
                $id_reunion = $multa['id_reunion'];
                $monto_pago = $multa['monto'];

                // Verificar que la reunión existe y el padre está registrado
                $verificar = DB::select(
                    "SELECT
                    reunion_padres.id_reunion_padres,
                    reunion_padres.id_reunion,
                    reunion_padres.id_padre_familia,
                    reunion_padres.asistencia,
                    reunion_padres.detalles,
                    reunion.titulo,
                    reunion.multa_precio
                    FROM
                    reunion_padres
                    INNER JOIN reunion ON reunion_padres.id_reunion = reunion.id_reunion
                    WHERE reunion_padres.id_reunion = ? AND reunion_padres.id_padre_familia = ?
                    ",
                    [$id_reunion, $id_padre]
                );

                if (empty($verificar)) {
                    DB::rollBack();
                    return back()->with("INCORRECTO", "Error: Una de las reuniones seleccionadas no está registrada para este padre de familia");
                }

                $titulo = $verificar[0]->titulo;
                $id_reunion_padres = $verificar[0]->id_reunion_padres;
                $asistencia = $verificar[0]->asistencia;
                $detalles = $verificar[0]->detalles;

                // Verificar si ya pagó la multa
                if ($asistencia != null || $detalles != null) {
                    DB::rollBack();
                    return back()->with("INCORRECTO", "Error: La multa de la reunión '{$titulo}' ya está pagada o el padre asistió");
                }

                // Actualizar reunion_padres para marcar la multa como pagada
                $updateResult = DB::update(
                    "UPDATE reunion_padres SET detalles = ?, monto_aportado = ? WHERE id_reunion = ? AND id_padre_familia = ?",
                    ["MULTA PAGADA", $monto_pago, $id_reunion, $id_padre]
                );

                if (!$updateResult) {
                    DB::rollBack();
                    return back()->with("INCORRECTO", "Error al actualizar el registro de la reunión {$titulo}");
                }

                // Registrar el pago en la tabla pago
                $insertResult = DB::insert(
                    "INSERT INTO pago(id_padre_familia, id_usuario, pago_concepto, monto_pago, debe, fecha_pago_boucher, numero_pago_boucher, foto_boucher, comentario, id_reunion_padres, estado) 
                    VALUES(?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)",
                    [$id_padre, $id_usuario, "MULTA REUNION: $titulo", $monto_pago, 0, $fecha_pago, $numero_recibo, $nombre_foto, $comentarios, $id_reunion_padres, 'VIGENTE']
                );

                if (!$insertResult) {
                    DB::rollBack();
                    return back()->with("INCORRECTO", "Error al registrar el pago de la multa para {$titulo}");
                }

                // Guardar el ID del pago
                $id_pagos[] = DB::getPdo()->lastInsertId();
            }

            // Registrar en historial_pagos
            $usuario = auth()->user()->nombres;
            $totalPagado = array_sum(array_map(function ($id) {
                $pago = DB::select("SELECT monto_pago FROM pago WHERE id_pago = ?", [$id]);
                return $pago[0]->monto_pago ?? 0;
            }, $id_pagos));

            $descripcion = "El usuario '{$usuario}' ha registrado " . count($id_pagos) . " pagos múltiples de multas" .
                ($numero_recibo ? " con el número de operación {$numero_recibo}" : "") .
                " por un total de S/. " . number_format($totalPagado, 2);

            DB::insert(
                "INSERT INTO historial_pagos (id_padre_familia, descripcion, registrado_por, fecha) VALUES (?, ?, ?, ?)",
                [$id_padre, $descripcion, $usuario, now()]
            );

            DB::commit();

            // Array asociativo para mapear cada id_pago con su comentario específico
            $comentarios_array = array_fill_keys($id_pagos, $comentarios);

            // Guardar un mensaje de éxito en la sesión
            return redirect()->back()->with([
                'id_pagos' => $id_pagos,
                'comentarios' => $comentarios_array,
            ]);
        } catch (\Illuminate\Validation\ValidationException $e) {
            // Manejar errores de validación
            return back()->withErrors($e->validator)->withInput()->with("INCORRECTO", "Por favor corrija los errores en el formulario");
        } catch (\Exception $e) {
            // Manejar cualquier otra excepción
            DB::rollBack();
            return back()->with("INCORRECTO", "Error al registrar los pagos: " . $e->getMessage())->withInput();
        }
    }

    public function edit($id_pago)
    {
        // Obtener detalles del pago
        $pago = DB::select("
            SELECT 
                p.*,
                u.nombres as nombre_usuario,
                pf.padre_nombres,
                pf.padre_ape_pat,
                pf.padre_ape_mat
            FROM pago p
            INNER JOIN usuario u ON p.id_usuario = u.id_usuario
            INNER JOIN padre_familia pf ON p.id_padre_familia = pf.id_padre_familia
            WHERE p.id_pago = ?
        ", [$id_pago]);

        if (empty($pago)) {
            return back()->with("INCORRECTO", "El pago no existe");
        }

        // Determinar si es un pago por reunión
        $esPagoReunion = !empty($pago[0]->id_reunion_padres);

        // Obtener límite máximo para el monto si es un pago de aporte
        $montoMaximo = null;
        if (!$esPagoReunion && !empty($pago[0]->id_aporte_padres)) {
            $aportePadre = DB::select("
                SELECT 
                    ap.*,
                    (ap.monto_aporte - (ap.monto_aportado - ?)) as monto_maximo_disponible
                FROM 
                    aporte_padres ap
                WHERE 
                    ap.id_aporte_padres = ?
            ", [$pago[0]->monto_pago, $pago[0]->id_aporte_padres]);

            if (!empty($aportePadre)) {
                // Si nunca ha aportado, el máximo es el monto total del aporte
                // Si ya aportó algo, el máximo es el valor "debe" + el monto actual del pago
                $montoMaximo = $aportePadre[0]->monto_maximo_disponible;
            }
        }

        return view("vistas/pagos.edit", compact("pago", "esPagoReunion", "montoMaximo"));
    }

    public function update(Request $request, $id_pago)
    {
        // Validación según el tipo de pago
        if ($request->has('tipo_pago') && $request->tipo_pago === 'reunion') {
            $request->validate([
                "comentario" => "nullable|string|max:255",
                "boucher" => "nullable|file|max:2048|mimes:jpeg,png,jpg,gif,pdf",
            ]);
        } else {
            $request->validate([
                "monto_pago" => "required|numeric",
                "comentario" => "nullable|string|max:255",
                "boucher" => "nullable|file|max:2048|mimes:jpeg,png,jpg,gif,pdf",
            ]);
        }

        // Obtener información del pago actual
        $pagoActual = DB::select("SELECT * FROM pago WHERE id_pago = ?", [$id_pago]);

        if (empty($pagoActual)) {
            return back()->with("INCORRECTO", "El pago no existe");
        }

        $pagoActual = $pagoActual[0];
        $montoAnterior = $pagoActual->monto_pago;

        // Determinar el monto nuevo según el tipo de pago
        if ($pagoActual->id_reunion_padres) {
            // Si es pago de reunión, no permitimos cambiar el monto
            $montoNuevo = $montoAnterior;
        } else {
            $montoNuevo = $request->monto_pago;

            // Validar que el monto nuevo no exceda el máximo permitido
            if ($pagoActual->id_aporte_padres) {
                $aportePadre = DB::select("
                    SELECT 
                        ap.*,
                        (ap.monto_aporte - (ap.monto_aportado - ?)) as monto_maximo_disponible
                    FROM 
                        aporte_padres ap
                    WHERE 
                        ap.id_aporte_padres = ?
                ", [$montoAnterior, $pagoActual->id_aporte_padres]);

                if (!empty($aportePadre) && $montoNuevo > $aportePadre[0]->monto_maximo_disponible) {
                    return back()->with("INCORRECTO", "El monto ingresado excede el máximo permitido de S/. " . $aportePadre[0]->monto_maximo_disponible);
                }
            }
        }

        $diferencia = $montoNuevo - $montoAnterior;

        // Manejo del archivo boucher
        $nombreBoucher = $pagoActual->foto_boucher;
        if ($request->hasFile('boucher')) {
            try {
                $archivo = $request->file('boucher');

                // Verificar que el archivo es válido
                if ($archivo->isValid()) {
                    // Eliminar boucher anterior si existe
                    if ($nombreBoucher && file_exists(public_path('storage/ARCHIVOS/pagos/' . $nombreBoucher))) {
                        unlink(public_path('storage/ARCHIVOS/pagos/' . $nombreBoucher));
                    }

                    $nombreBoucher = time() . '_' . $archivo->getClientOriginalName();

                    // Crear la carpeta si no existe
                    $directorio = public_path('storage/ARCHIVOS/pagos');
                    if (!file_exists($directorio)) {
                        mkdir($directorio, 0777, true);
                    }

                    // Mover el archivo
                    if (move_uploaded_file($archivo->getPathname(), $directorio . '/' . $nombreBoucher)) {
                        // El archivo se movió correctamente
                    } else {
                        return back()->with("INCORRECTO", "Error al mover el archivo");
                    }
                } else {
                    return back()->with("INCORRECTO", "El archivo no es válido");
                }
            } catch (\Exception $e) {
                return back()->with("INCORRECTO", "Error al subir el boucher: " . $e->getMessage());
            }
        }

        // Iniciar transacción
        DB::beginTransaction();

        try {
            // Actualizar las tablas correspondientes según el tipo de pago
            if ($pagoActual->id_aporte_padres) {
                // Es un pago de aporte - permitimos modificar el monto
                $aportePadre = DB::select("SELECT * FROM aporte_padres WHERE id_aporte_padres = ?", [$pagoActual->id_aporte_padres])[0];

                // Calcular nuevo monto aportado y deuda
                $nuevoMontoAportado = $aportePadre->monto_aportado - $montoAnterior + $montoNuevo;
                $nuevaDeuda = $aportePadre->monto_aporte - $nuevoMontoAportado;

                // Actualizar aporte_padres
                DB::update(
                    "UPDATE aporte_padres SET monto_aportado = ?, debe = ? WHERE id_aporte_padres = ?",
                    [$nuevoMontoAportado, $nuevaDeuda, $pagoActual->id_aporte_padres]
                );
            } elseif ($pagoActual->id_reunion_padres) {
                // Es un pago de multa de reunión - no se modifica el monto
                // Solo actualizamos si el monto es 0 (anulación)
                if ($montoNuevo == 0) {
                    DB::update(
                        "UPDATE reunion_padres SET detalles = NULL WHERE id_reunion_padres = ?",
                        [$pagoActual->id_reunion_padres]
                    );
                }
            }

            // Actualizar el registro del pago
            DB::update(
                "UPDATE pago SET monto_pago = ?, comentario = ?, foto_boucher = ?, estado = 'EDITADO' WHERE id_pago = ?",
                [$montoNuevo, $request->comentario, $nombreBoucher, $id_pago]
            );

            // Registrar en historial_pagos
            $usuario = auth()->user()->nombres;
            // Modificar el mensaje del historial según tipo de pago
            if ($pagoActual->id_reunion_padres) {
                $descripcion = "El usuario '{$usuario}' ha editado el pago #{$id_pago} (multa de reunión). Solo se modificaron comentarios o boucher.";
            } else {
                $descripcion = "El usuario '{$usuario}' ha editado el pago #{$id_pago}. Cambió el monto de S/. {$montoAnterior} a S/. {$montoNuevo}";
            }

            DB::insert(
                "INSERT INTO historial_pagos (id_padre_familia, descripcion, registrado_por, fecha) VALUES (?, ?, ?, ?)",
                [$pagoActual->id_padre_familia, $descripcion, $usuario, now()]
            );

            DB::commit();

            // Obtener el DNI del padre para redirigir correctamente
            $padre = DB::select("SELECT pf.padre_dni FROM padre_familia pf WHERE pf.id_padre_familia = ?", [$pagoActual->id_padre_familia]);

            if (empty($padre)) {
                return back()->with("CORRECTO", "Pago actualizado correctamente");
            }

            return redirect()->route('pagos.show', $padre[0]->padre_dni)->with("CORRECTO", "Pago actualizado correctamente");
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with("INCORRECTO", "Error al actualizar el pago: " . $e->getMessage());
        }
    }

    public function anular($id_pago)
    {
        // Obtener información completa del pago con todos los campos necesarios
        $pago = DB::select("
            SELECT p.*, pf.padre_dni 
            FROM pago p 
            INNER JOIN padre_familia pf ON p.id_padre_familia = pf.id_padre_familia 
            WHERE p.id_pago = ?
        ", [$id_pago]);

        if (empty($pago)) {
            return back()->with("INCORRECTO", "El pago no existe");
        }

        $pago = $pago[0];
        $dni_padre = $pago->padre_dni;

        // Verificar si el pago ya fue anulado
        if ($pago->monto_pago == 0) {
            return back()->with("INCORRECTO", "Este pago ya fue anulado anteriormente");
        }

        // Iniciar transacción
        DB::beginTransaction();

        try {
            // Revertir cambios según el tipo de pago
            if ($pago->id_aporte_padres) {
                // Es un pago de aporte
                $aportePadre = DB::select("
                    SELECT * FROM aporte_padres 
                    WHERE id_aporte_padres = ?
                ", [$pago->id_aporte_padres]);

                if (empty($aportePadre)) {
                    throw new \Exception("No se encontró el registro de aporte correspondiente");
                }

                $aportePadre = $aportePadre[0];

                // Calcular nuevo monto aportado y deuda
                $nuevoMontoAportado = $aportePadre->monto_aportado - $pago->monto_pago;
                if ($nuevoMontoAportado < 0) $nuevoMontoAportado = 0;

                $nuevaDeuda = $aportePadre->monto_aporte - $nuevoMontoAportado;

                // Actualizar aporte_padres - Asegurarse que se actualice correctamente
                $actualizacionAporte = DB::update(
                    "UPDATE aporte_padres SET monto_aportado = ?, debe = ? WHERE id_aporte_padres = ?",
                    [$nuevoMontoAportado, $nuevaDeuda, $pago->id_aporte_padres]
                );

                if ($actualizacionAporte != 1) {
                    throw new \Exception("No se pudo actualizar el registro en aporte_padres");
                }
            } elseif ($pago->id_reunion_padres) {
                // Es un pago de multa de reunión
                $reunionPadre = DB::select("
                    SELECT * FROM reunion_padres 
                    WHERE id_reunion_padres = ?
                ", [$pago->id_reunion_padres]);

                if (empty($reunionPadre)) {
                    throw new \Exception("No se encontró el registro de reunión correspondiente");
                }

                // Actualizar reunion_padres - Asegurarse de vaciar la columna detalles
                $actualizacionReunion = DB::update(
                    "UPDATE reunion_padres SET detalles = NULL WHERE id_reunion_padres = ?",
                    [$pago->id_reunion_padres]
                );

                if ($actualizacionReunion != 1) {
                    throw new \Exception("No se pudo actualizar el registro en reunion_padres");
                }
            } else {
                throw new \Exception("El pago no está asociado a ningún aporte o reunión");
            }

            // Eliminar boucher si existe
            if ($pago->foto_boucher && file_exists(public_path('storage/ARCHIVOS/pagos/' . $pago->foto_boucher))) {
                unlink(public_path('storage/ARCHIVOS/pagos/' . $pago->foto_boucher));
            }

            // Registrar en historial_pagos
            $usuario = auth()->user()->nombres;
            $descripcion = "El usuario '{$usuario}' ha anulado el pago #{$id_pago} por un monto de S/. {$pago->monto_pago}. Concepto: {$pago->pago_concepto}";

            DB::insert(
                "INSERT INTO historial_pagos (id_padre_familia, descripcion, registrado_por, fecha) VALUES (?, ?, ?, ?)",
                [$pago->id_padre_familia, $descripcion, $usuario, now()]
            );

            // Marcar el pago como anulado (monto_pago = 0, estado = ANULADO y agregar comentario)
            DB::update(
                "UPDATE pago SET estado = 'ANULADO', comentario = CONCAT(IFNULL(comentario, ''), ' [ANULADO: ', NOW(), ' por {$usuario}]') WHERE id_pago = ?",
                [$id_pago]
            );

            DB::commit();
            return redirect()->route('pagos.show', $dni_padre)->with("CORRECTO", "Pago anulado correctamente");
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with("INCORRECTO", "Error al anular el pago: " . $e->getMessage());
        }
    }

    public function limpiarHistorial($id_padre_familia = null)
    {
        try {
            // Eliminar todos los registros de la tabla historial_pagos
            $eliminados = DB::delete("DELETE FROM historial_pagos where id_padre_familia = ?", [$id_padre_familia]);

            return back()->with("CORRECTO", "Se han eliminado todos los registros del historial de pagos");
        } catch (\Exception $e) {
            return back()->with("INCORRECTO", "Error al eliminar el historial: " . $e->getMessage());
        }
    }
}
